//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.12

/*!
\qmltype     UiRadioButtonGroup
\inqmlmodule we.captivo.Ui
\since       we.captivo.Ui 1.0
\inherits    QtQuick2::Column
\ingroup     captivo-ui-controls

\brief A column of grouped radio buttons.

The UiRadioButtonColumn layouts several UiRadioButton controls into a column
and automatically assigns a UiRadioButtonGroup to the buttons:

\qml
import QtQuick 2.4

import wesual.Ui 1.0

UiRadioButtonColumn {
    UiRadioButton {
        text : "Option 1"
    }
    UiRadioButton {
        text : "Option 2"
    }
    UiRadioButton {
        text : "Option 3"
    }
}
\endqml
\sa UiRadioButton UiRadioButtonGroup
*/

Column {
    id : uiRadioButtonColumn

    /*!
    \qmlproperty list<UiRadioButton> UiRadioButtonColumn::buttons
    \default
    \brief The list of radio buttons to group together.
    */
    default property alias buttons : buttonGroup.buttons

    /*!
    \brief The index of the currently checked radio button.

    The checked button can be changed by assigning a valid button index
    to this property.
    */
    property int currentIndex : -1

    children : buttons
    spacing  : 9

    baselineOffset : {
        if (buttons.length > 0) {
            return buttons[0].y + buttons[0].baselineOffset;
        }

        return 0;
    }

    Component.onCompleted : buttonGroup.updateButtions()
    onCurrentIndexChanged : {
        if (currentIndex < -1 || currentIndex >= buttons.length) {
            buttonGroup.assignIndex();

            return;
        }

        try {
            buttonGroup.activeButton = buttonGroup.buttons[currentIndex];
        } catch (e) {
            buttonGroup.activeButton = null;
        }
    }

    data : UiRadioButtonGroup {
        id : buttonGroup

        property list<UiRadioButton> buttons

        function assignIndex() {
            var index = -1;

            for (var i = 0; i < buttons.length; ++i) {
                if (buttons[i] === activeButton) {
                    index = i;
                    break;
                }
            }

            uiRadioButtonColumn.currentIndex = index
        }

        function updateButtions() {
            for (var i = 0; i < buttons.length; ++i) {
                buttons[i].group = buttonGroup;
            }
        }

        onButtonsChanged      : updateButtions()
        onActiveButtonChanged : assignIndex()
    }
}
